#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
A delphi building utility: generic build task execution.

Require
  Python        2.3

Classes
  class         taskHandler(object aTask, object logger)
                  extends py_netasq.commonlib.asqCommand::cmdHandler
                  extends py_netasq.commonlib.asqWin::winCmdHandler
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.18 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqCommand, asqPath, asqWin

from    py_netasq               import building as asqBuild

import  taskItem                                as asqTaskItem
import  errors


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

# cross platform !
# The best command handler will be picked up according to the host system
if -1 <> asqWin.getWindowsPlatform :
  __parentClass = asqWin.winCmdHandler
else :
  __parentClass = asqCommand.cmdHandler

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class taskHandler(__parentClass) :
    """\
      Simple object to handle tasks execution.
      
      constructor       taskHandler(object aTask)
      
      var               run(object aTask, bool raw, *arglist, **argdict)
      
      property object   task
    """

    # set inner task class
    taskClass = asqTaskItem.taskItem

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, aTask=None) :
        """\
          constructor taskHandler(object aTask)
          
          Create a new object intended to ease command line execution.
          
          param aTask                 task to execute.
                                       defaults to None
        """
        super(taskHandler, self).__init__(None)
        # working dir. stack
        self._cwdStack  = list()
        # path to temporary file, None => no temp file
        self._tmpFile   = None
        # emit event handler
        self._onEmitMsg = None
        
        # creates a 'taskClass' object
        self._task      = self.taskClass(False)
        # use setter method - self._task.assign(aTask)
        self.task       = aTask


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _debugKeyVal(self, key, val) :
        """\
          void _debugKeyVal(string key, var val)
          
          Formats and prints out the given key/value duo. Ensures
          display consistency.
          
          param key                   value name
          param val                   value to display
        """
        asqBuild.logger.debug('%-15s %s', key, repr(val))


    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        self._debugKeyVal('curr. work dir', os.getcwd())
        asqBuild.logger.incDepthDebug()
        for i in xrange(len(self._cwdStack)-1, -1, -1) :
          self._debugKeyVal('_cwdStack[%d]' % (i, ), self._cwdStack[i])
        asqBuild.logger.decDepthDebug()
        asqBuild.logger.debug('* task.debug()')
        asqBuild.logger.incDepthDebug()
        self._task.debug()
        asqBuild.logger.decDepthDebug()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def run(self, aTask=None, raw=False, *arglist, **argdict) :
        """\
          var run(object aTask, bool raw, *arglist, **argdict)
          
          Execute inner task line, if "task" is given, it will replace
          the current taskItem instance; The shell status is returned,
          'None' if execution was succesful, any other value means
          failure. If "raw" is set to True, then no peculiar operation
          should be done upon the command's output.
          
          Note : 'taskError' or 'taskHandlerError' exceptions may be
           thrown by this method.
          
          param aTask                 taskItem to execute.
                                       defaults to None
          param raw                   disable output formatting
                                        defaults to False
          param *arglist              extra parameters.
          param **argdict             extra named parameters.
          return                      shell status
        """
        asqBuild.logger.trace('taskHandler.run(aTask, %s)', str(raw))
        asqBuild.logger.incDepthTrace()
        try :
          
          if aTask is not None :
            self.task = aTask
          
          self._emitMsg(self.task.antDisplay())
          
          if not self._task.enabled :
            return None
          elif not self._task.isValid() :
            raise errors.taskError('non valid task', self._task)
          
          buffer = self._task.toCmdLine()
          # self._task.toCmdLine() SHOULD NOT return None
          assert buffer is not None
          
          # a taskHandlerError could be raised...
          result = super(taskHandler, self).run(buffer, raw, *arglist, **argdict)
          
          # once _teardown() has been called, the working directory stack
          # should be empty
          assert 0 == len(self._cwdStack)                                , \
                'working directory stack is not empty !'
          
          return result
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setup(self, *arglist, **argdict) :
        """\
          void _setup(*arglist, **argdict)
          
          Called right before the actual command execution; should be
          overidden to set up the execution environment.
          
          - change current working directory to task's working dir.
          
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        asqBuild.logger.trace('taskHandler._setup()')
        asqBuild.logger.incDepthTrace()
        try :
          
          super(taskHandler, self)._setup(*arglist, **argdict)
          self._pushCwd(self._task.workdir)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _teardown(self, *arglist, **argdict) :
        """\
          void _teardown(*arglist, **argdict)
          
          Called right after the actual command execution; should be
          overidden to clean up the execution environment.
          
          - change back current working directory
          
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
        """
        asqBuild.logger.trace('taskHandler._teardown()')
        asqBuild.logger.incDepthTrace()
        try :
          
          self._popCwd()
          super(taskHandler, self)._teardown(*arglist, **argdict)
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _execute(self, cmdStr, raw, *arglist, **argdict) :
        """\
          var _execute(string cmdStr, bool raw, *arglist, **argdict)
          
          Execute inner command line and return the shell status.
          Each line from stdout and stderr will be decoded and processed,
          i.e. lines will go through _decodeOutput() and _processOutput().
          
          param cmdStr                command as a string to execute.
          param raw                   disable output formatting
          param *arglist              extra parameters list.
          param **argdict             extra named parameters.
          return                      shell status
        """
        asqBuild.logger.trace('taskHandler._execute(%s, %s)', cmdStr, str(raw))
        asqBuild.logger.incDepthTrace()
        try :
          
          status = super(taskHandler, self)._execute(cmdStr, raw, *arglist, **argdict)
          if status is None :
            return None
          else :
            raise errors.taskHandlerError(status, self.output, self._task)
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _createTmpFile(self, suffix="", dir=None) :
        """\
          void _createTmpFile(string suffix, string dir)
          
          Creates a temporary file in the most secure manner possible.
          No dot will be automatically put between the file name and
          the suffix, it should be added manually.
          Self._tmpFile is used to held the result of absolute path of the
          file. use mkstemp(). New in version 2.3.
          
          param suffix                file name suffix
                                        defaults to ""
          param dir                   parent dir of temp file
                                        defaults to None
        """
        asqBuild.logger.trace('taskHandler._createTmpFile(%s, %s)', suffix, str(dir))
        asqBuild.logger.incDepthTrace()
        try :
          
          self._tmpFile = asqPath.makeTmpFile(suffix=suffix, dir=dir)
          assert os.path.isabs(self._tmpFile)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _deleteTmpFile(self) :
        """\
          void _deleteTmpFile(void)
          
          Deletes a previously created temporary file.
        """
        asqBuild.logger.trace('taskHandler._deleteTmpFile()')
        asqBuild.logger.incDepthTrace()
        try :
          
          pass
          # Comment out the following code lines to keep temporary files
          #  This is useful to understand why a build did fail, especially
          #  when dealing with Setup Factory errors...
          if self._tmpFile :
            asqPath.delete(self._tmpFile, True)
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _pushCwd(self, aPath) :
        """\
          void _pushCwd(string aPath)
          
          Changes current working directory, the previous working dir.
          path is appended on top of the inner directory stack.
          If given path is None, then the working dir. won't be changed,
          but a None value will be pushed into the stack.
          
          param aPath                 new working directory.
        """
        asqBuild.logger.trace('taskHandler._pushCwd(%s)', str(aPath))
        asqBuild.logger.incDepthTrace()
        try :
          
          if aPath is None :
            self._cwdStack.append(None)
          else :
            buffer = os.getcwd()
            asqPath.changeDir(aPath)
            self._cwdStack.append(buffer)
         
        finally :
          asqBuild.logger.decDepthTrace()


    def _popCwd(self) :
        """\
          void _popCwd(void)
          
          Restores previous working directory.
        """
        asqBuild.logger.trace('taskHandler._popCwd()')
        asqBuild.logger.incDepthTrace()
        try :
          
          if not self._cwdStack :
            return
          buffer = self._cwdStack.pop()
          if buffer is not None :
            os.chdir(buffer)
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _emitMsg(self, message) :
        """\
          void _emitMsg(string message)
          
          Sends out given message to listeners
        """
        if callable(self._onEmitMsg) :
          self._onEmitMsg(message)


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getOnEmitMsg(self) :
        """\
          object _getOnEmitMsg(void)
          
          Getter - onEmitMsg property
          Returns handler bound to the emitMsg event
          
          return                      onEmitMsg handler
        """
        return self._onEmitMsg


    def _getTask(self) :
        """\
          object _getTask(void)
          
          Getter - task property
          Returns the current task object. Can't be None.
          
          return                      current task object
        """
        return self._task


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setOnEmitMsg(self, value) :
        """\
          void _setOnEmitMsg(object value)
          
          Setter - onEmitMsg property
          Sets the handler bound to the emitMsg event
          
          param value                 event handler
        """
        if not callable(value) :
          self._onEmitMsg = None
        else :
          self._onEmitMsg = value


    def _setTask(self, value) :
        """\
          void _setTask(object value)
          
          Setter - task property
          Sets the current task
          
          param value                 task object
        """
        self._task.assign(value)
        #~ self._task.completePaths()


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    task        = property(
        doc  = "task item to handle",
        fget = _getTask,
        fset = _setTask,
        fdel = None
      )

    onEmitMsg   = property(
        doc  = "trigerred each time a message is emitted",
        fget = _getOnEmitMsg,
        fset = _setOnEmitMsg,
        fdel = None
      )


  # /class taskHandler  - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -