#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Foundation class of all building task classes : taskItem,
 taskList and the inherited classes.

Require
  Python        2.2

Classes
  class         task(bool enabled)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.33 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  os

from    py_netasq.commonlib     import asqMapping, asqPath, asqSequence
from    py_netasq.commonlib     import asqTypes

from    py_netasq               import building as asqBuild

import  checkResult                             as asqCheckResult


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class task(object) :
    """\
      Foundation class of all building task classes (taskItem and taskList).
      For single task items, there is a huge difference between the 'None'
      value and the empty string; each attribute that is equal to 'None',
      means "use taskList value for that attribute", whereas empty string
      does override tasklist values, meaning "no value for that attribute".
      
      
      constructor       task(bool enabled)
     
      void              debug(void)
      object            check(object status)
     
      void              fromDict(list aDict, string defaultType)
      list              toDict(bool filtered, bool strict)
     
      void              fromList(list aList, string defaultType)
      list              toList(bool filtered, bool strict)
      
      object            clone(void)
      void              assign(object source)
     
      void              completePaths(var baseDir)
      
      void              normalize(void)
      void              reset(void)
      
      void              beginUpdate(void)
      void              endUpdate(void)
      
      bool              isValid(void)
      bool              isCompatibleWith(string aType)
    """

    # class task type, nuff said :)
    classTaskType = 'task'

    # task compatibility
    compatibility = (
        # none :)
      )

    # Properties defined for that task type and its descendant.
    # Note : attribute names should be "lower case" only.
    # Note : Since task is the mother class, those properties
    #  _should_ be shared by all task items (actually child classes
    #  may overwrite it)
    attributes    = (
        'taskname',             # task name (should held section name)
        'tasktype',             # task type as a string
        'enabled',              # is task active ?
        'workdir',              # task base directory
        'comments',             # about this task
        'others'                # unrecognized attributes
      )


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, enabled) :
        """\
          constructor task(bool enabled)
          
          A single task object; the 'attributes' property defines the
          available object's properties.
          e.g. if aTask.attributes = ('logFile', 'comments') then
            aTask.logFile and aTask.comments should be available.
            
          param enabled               is task enabled
        """
        asqBuild.logger.trace('task.__init__(%s)', enabled)
        asqBuild.logger.incDepthTrace()
        try :
          
          super(task, self).__init__()
          self.__updating = 0
          self.reset()
          # triggers ON PURPOSE task normalization
          self.enabled = enabled
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __setattr__(self, name, value) :
        """\
          void __setattr__(string name, var value)
          
          Special method - Called when an attribute assignment is attempted.
          This is called instead of the normal mechanism (i.e. store the
          value in the instance dictionary). name is the attribute name,
          value is the value to be assigned to it.
          
          If name is not found in defined attributes list, and is not a
          mangled name (_classname__name), then (name, value) is added to
          the object's "others" dictionnary.
          
          Each time an attribute value is set, 'normalize()' should be
          called to ensure consistency.
          
          param name                  attribute to set
          param value                 new value
        """
        asqBuild.logger.trace('task.__setattr__(%s, %s)', name, repr(value))
        asqBuild.logger.incDepthTrace()
        try :
          
          if name.startswith('_') :
            object.__setattr__(self, name, value)
          elif name not in self.attributes :
            self.others[name] = value
          else :
            object.__setattr__(self, name, value)
            if 0 == self.__updating :
              self.normalize()
          
        finally :
          asqBuild.logger.decDepthTrace()


    def __nonzero__(self) :
        """\
          bool __nonzero__(void)
          
          Called to implement truth value testing, and the built-in operation
          bool(); should return False or True, or their integer equivalents
          0 or 1.
          Tells whether or not the task is "empty": an empty task has no
          value defined (None values) for its attributes. Performed right
          after a "task.reset()" this function should return "False".
          
          return                      is task not empty
        """
        asqBuild.logger.trace('task.__nonzero__()')
        asqBuild.logger.incDepthTrace()
        try :
          
          for key in self.attributes :
            asqBuild.logger.trace('self._attributeNonZero(%s)', key)
            if self._attributeNonZero(key) :
              return True
          else :
            return False
          
        finally :
          asqBuild.logger.decDepthTrace()


    def __str__(self) :
        """\
          string __str__(void)
          
          Called by the str() built-in function and by the print statement
          to compute the "informal" string representation of an object. This
          differs from __repr__() in that it does not have to be a valid
          Python expression: a more convenient or concise representation may
          be used instead. The return value must be a string object
          
          Returns a pseudo command line: arguments are shortened to reduce
          information cluttering. If current task is found non valid, then
          an empty string will be returned.
          
          return                      pseudo command line string
        """
        return ''


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _debugKeyVal(self, key, val) :
        """\
          void _debugKeyVal(string key, var val)
          
          Formats and prints out the given key/value duo. Ensures
          display consistency.
          
          param key                   value name
          param val                   value to display
        """
        asqBuild.logger.debug('%-15s %s', key, repr(val))


    def debug(self) :
        """\
          void debug(void)
          
          Prints out the main attributes values through the logging
          utility.
        """
        self._debugKeyVal('bool(self)'    , bool(self))
        self._debugKeyVal('self.isValid()', self.isValid())
        asqBuild.logger.debug('--')
        for key in self.attributes :
          asqBuild.logger.trace('self._attributeDebug(%s)', key)
          buffer = self._attributeDebug(key)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _attributeNonZero(self, name) :
        """\
          bool _attributeNonZero(string name)
          
          Callback used by self.__nonzero__()
          Checks that attribute is not empty.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute NOT empty ?
        """
        assert isinstance(name, basestring)                             , \
              'given "name" is not a string value'
        assert name in self.__dict__                                    , \
              '%s missing from object\'s properties' % (name, )
        
        # 'taskname', 'tasktype' and 'enabled' does not matter here
        if name in ('taskname', 'tasktype', 'enabled', ) :
          return False
        elif name in ('others', ) :
          return bool(self.others)
        else :
          return self.__dict__[name] is not None


    def _attributeDebug(self, name) :
        """\
          void _attributeDebug(string name)
          
          Callback used by self.debug()
          Prints out debug information for given attribute.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if 'others' != name :
          self._debugKeyVal(name, self.__dict__[name])
        elif not self.others :
          # empty 'others' dictionnary
          self._debugKeyVal(name, self.__dict__[name])
        else :
          asqBuild.logger.debug(name)
          asqBuild.logger.incDepthDebug()
          for key in asqSequence.sort(None, self.__dict__[name].keys()) :
            self._debugKeyVal(key, self.__dict__[name][key])
          asqBuild.logger.decDepthDebug()


    def _attributeCheck(self, status, name) :
        """\
          void _attributeCheck(object status, string name)
          
          Callback used by self.check(...)
          Checks the value of attribute; Appends error or warning
          messages to given check status.
          Should be overriden to set the behavior for children classes.
          
          param status                checkResult object
          param name                  attribute name
        """
        if 'workdir' == name :
          self._checkDirPath(status, name, 'working directory')
        elif 'others' == name :
          if 'logfile' in self.others :
            status.addMsgWarning('%-15s deprecated property' % ('logfile', ))


    def _attributeToList(self, aList, name) :
        """\
          void _attributeToList(list aList, string name)
          
          Callback used by self.toList(...)
          Appends attribute to given "aList" list.
          Should be overriden to set the behavior for children classes.
          
          param aList                 list of attributes
          param name                  attribute name
        """
        if 'others' == name  :
          aList.extend( self.others.items() )
        else :
          aList.append( (name, self.__dict__[name]) )


    def _attributeNormalize(self, name) :
        """\
          void _attributeNormalize(string name)
          
          Callback used by self.normalize()
          Normalizes attribute : ensures that its type is correct.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if   'tasktype' == name :
          if self.classTaskType != self.tasktype :
            object.__setattr__(self, 'tasktype', self.classTaskType)
        elif 'others'   == name :
          if not isinstance(self.others, dict) :
            object.__setattr__(self, 'others', dict())
        elif 'workdir'  == name :
          if not asqTypes.isNonEmptyInstance(self.workdir, basestring) :
            object.__setattr__(self, 'workdir', '.')
        elif 'enabled'  == name :
          if isinstance(self.enabled, bool) :
            pass
          elif not isinstance(self.enabled, basestring) :
            object.__setattr__(self, 'enabled', bool(self.enabled))
          else :
            buffer = asqTypes.asBoolean(self.enabled)
            object.__setattr__(self, 'enabled', buffer)


    def _attributeReset(self, name) :
        """\
          void _attributeReset(string name)
          
          Callback used by self.reset()
          Resets attribute value.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
        """
        if   'tasktype' == name :
          object.__setattr__(self, 'tasktype', self.classTaskType)
        elif 'enabled'  == name :
          object.__setattr__(self, 'enabled' , False)
        elif 'others'   != name : # default set attribute value to None
          object.__setattr__(self, name      , None)
        else :
          if not hasattr(self, 'others') :
            object.__setattr__(self, 'others', dict())
          else :
            self.others.clear()


    def _attributeIsPath(self, name) :
        """\
          bool _attributeIsPath(string name)
          
          Callback used by self.completePaths(...)
          Returns True if attribute is a path, please note that the
          'workdir' attribute should NOT be considered as a path here.
          Should be overriden to set the behavior for children classes.
          
          param name                  attribute name
          return                      is attribute a path
        """
        return False


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _checkValue(self, status, name, label, vital=False) :
        """\
          void _checkValue(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s %s value is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s %s value is not set' % (name, label))
        elif not isinstance(buffer, basestring) :
          status.addMsgError('%-15s %s value is invalid' % (name, label))
        elif not buffer : # empty string
          status.addMsgWarning('%-15s %s value is empty' % (name, label))


    def _checkDirPath(self, status, name, label, vital=False) :
        """\
          void _checkDirPath(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s path to %s is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s path to %s is not set' % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s path to %s is invalid' % (name, label))
        elif not os.path.isdir(buffer) :
          if vital :
            status.addMsgError('%-15s cannot find %s' % (name, label))
          else :
            status.addMsgWarning('%-15s cannot find %s' % (name, label))


    def _checkFilePath(self, status, name, label, vital=False) :
        """\
          void _checkFilePath(object status, string name, string label, bool vital)
          
          Helper function used by _attributeCheck(). Provided to reduce
          redundancy. If "vital" is set to True, then ANY abnormal setting
          will produce errors instead of warnings.
          
          param status                checkResult object
          param name                  attribute name
          param label                 attribute formal name
          param vital                 is attribute setting vital ?
                                        defaults to False
        """
        buffer = self.__dict__[name]
        if buffer is None :
          if vital :
            status.addMsgError('%-15s path to %s is not set' % (name, label))
          else :
            status.addMsgWarning('%-15s path to %s is not set' % (name, label))
        elif not asqTypes.isNonEmptyInstance(buffer, basestring) :
          status.addMsgError('%-15s path to %s is invalid' % (name, label))
        elif not os.path.isfile(buffer) :
          if vital :
            status.addMsgError('%-15s cannot find %s' % (name, label))
          else :
            status.addMsgWarning('%-15s cannot find %s' % (name, label))


    def check(self, status=None, baseDir=None) :
        """\
          object check(object status, string baseDir)
          
          Checks attribute values, fills given status with gathered error
          or warning messages, and then returns the 'status' object. If
          'status' is None, then a new "checkResult" object will be
          instanciated.
          
          Note : The check() should be done upon a merged task.
          
          param status                checkResult object
                                        Defaults to None
          param baseDir               base for absolute path
                                        Defaults to None
          return                      check result
        """
        if status is None :
          asqBuild.logger.trace('task.check(status=None)')
        else :
          asqBuild.logger.trace('task.check(status)')
        asqBuild.logger.incDepthTrace()
        try :
          
          if status is None :
            status = asqCheckResult.checkResult(header=self.taskname)
          if not self.enabled :
            status.addMsgInfo('task is not enabled')
          elif not bool(self) :
            status.addMsgWarning('task is empty')
          else :
            self.completePaths(baseDir)
            for key in self.attributes :
              asqBuild.logger.trace('self._attributeCheck(status, %s)', key)
              self._attributeCheck(status, key)
          return status
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def fromDict(self, aDict, defaultType=None) :
        """\
          void fromDict(dict aDict, string defaultType)
          
          Reset object's attributes and then overwrite attributes with those
          provided by given dictionnary. If there is no "tasktype" value
          found in dict., then 'defaultType' is assumed.
          If list is compatible with current object, then unrecognised
          attributes are added to the "others" dict. If not, then only
          the commonly shared attributes will be copied, all other
          attributes will simply be ignored.
          
          e.g.
            { 'comments': 'dummy comment', 'logfile': None, 'foo': 'bar', }
            => self.comments = 'dummy comment'
               self.others   = { 'logfile': None, 'foo' : 'bar' }
          
          param aDict                 attributes dictionnary
          param defaultType           assumed tasktype, defaults to None
        """
        asqBuild.logger.trace('task.fromDict(aDict, defaultType=%s)', defaultType)
        asqBuild.logger.incDepthTrace()
        try :
          
          # create a copy, avoid making unnecessary changes to original
          buffer = aDict.copy()
          # current tasktype value should not be overwritten
          tkType = asqMapping.extractItem(buffer, 'tasktype', defaultType)
          
          if not self.isCompatibleWith(tkType) :
            # attribute list may be reduced (!) by child class
            filter = asqSequence.intersection(task.attributes, self.attributes)
            asqMapping.filterKeys(buffer, filter, False)
            
          self.beginUpdate()
          try :
            self.reset()
            for (key, val) in buffer.iteritems() :
              self.__setattr__(key, val)
          finally :
            self.endUpdate()
            self.normalize()
            del buffer
          
        finally :
          asqBuild.logger.decDepthTrace()


    def toDict(self, filtered=False, strict=False) :
        """\
          list toDict(bool filtered, bool strict)
          
          Return object's attributes, as a dictionnary.
          If 'filtered' is set to True, then attribute values equals
          to "None" won't be added to dict.
          If 'strict' is set to True, then the "others" attribute will
          be skipped.
          
          Note : lists keep order, dictionnary don't
          
          e.g. self.comments = 'dummy comment'
               self.others  = { 'logfile': None, 'foo' : 'bar' }
            
              toDict(filtered=False)
            => { 'comments': 'dummy comment', 'logFile': None, 'foo': 'bar' }
              toDict(filtered=True)
            => { 'comments': 'dummy comment', 'foo': 'bar' }
            
          param filtered              skip None values
                                        Defaults to False
          param strict                exclude 'others' attribute
                                        Defaults to False
          return                      dict of attributes
        """
        asqBuild.logger.trace('task.toDict(filtered=%s, strict=%s)', filtered, strict)
        asqBuild.logger.incDepthTrace()
        try :
          
          result = self.toList(filtered, strict)
          try :
            return dict(result)
          finally :
            del result
          
        finally :
          asqBuild.logger.decDepthTrace()


    def fromList(self, aList, defaultType=None) :
        """\
          void fromList(list aList, string defaultType)
          
          Reset object's attributes and then overwrite attributes with those
          provided by given list (Note : list should be a valid mapping).
          If there is no "tasktype" value found in list, then 'defaultType'
          is assumed.
          If list is compatible with current object, then unrecognised
          attributes are added to the "others" dict. If not, then only
          the commonly shared attributes will be copied, all other
          attributes will simply be ignored.
          
          e.g. [('comments', 'dummy comment'), ('logFile', None), ('foo', 'bar')]
            => self.comments = 'dummy comment'
               self.others   = { 'logfile': None, 'foo' : 'bar' }
          
          param aList                 list of attributes
          param defaultType           assumed tasktype, defaults to None
        """
        asqBuild.logger.trace('task.fromList(aList, defaultType=%s)', defaultType)
        asqBuild.logger.incDepthTrace()
        try :
          
          buffer = dict(aList)
          try :
            self.fromDict(buffer, defaultType)
          finally :
            del buffer
          
        finally :
          asqBuild.logger.decDepthTrace()


    def toList(self, filtered=False, strict=False) :
        """\
          list toList(bool filtered, bool strict)
          
          Return object's attributes, using the following format 
            [(attrA, val1), (attrB, val2), ...]
          If 'filtered' is set to True, then attribute values equals
          to "None" won't be added to dict.
          If 'strict' is set to True, then the "others" attribute will
          be skipped.
          
          Note : lists keep order, dictionnary don't
          
          e.g. self.comments = 'dummy comment'
               self.others  = { 'logfile': None, 'foo' : 'bar' }
            
              toList(False)
            => [('comments', 'dummy comment'), ('logFile', None), ('foo', 'bar')]
              toList(True)
            => [('comments', 'dummy comment'), ('foo', 'bar')]
            
          param filtered              skip None values
                                        Defaults to False
          param strict                exclude 'others' attribute
                                        Defaults to False
          return                      list of attributes
        """
        asqBuild.logger.trace('task.toList(filtered=%s, strict=%s)', filtered, strict)
        asqBuild.logger.incDepthTrace()
        try :
          
          result = list()
          for key in self.attributes :
            if strict and 'others' == key :
              pass
            else :
              asqBuild.logger.trace('self._attributeToList(result, %s)', key)
              self._attributeToList(result, key)
          if filtered :
            result[:] = filter(lambda x : x[1] is not None, result)
          return result
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def clone(self, *arglist, **argdict) :
        """\
          object clone(void)
          
          Create a deep copy for current object; Alas, __deepcopy__() is
          somewhat a little hard to grasp.
          
          return                      object clone
        """
        result = self.__class__(self.enabled, *arglist, **argdict)
        result.assign(self)
        return result


    def assign(self, source) :
        """\
          void assign(object source)
          
          Copies task item's attributes from 'source' task item.
          
          param source                source task item
        """
        if not isinstance(source, task) :
          self.reset()
        else :
          self.fromList(source.toList(False))


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _completePath(self, path, baseDir) :
        """\
          var _completePath(string path, string baseDir)
          
          
          Completes given path, using 'baseDir' value as base dir (thus the
          name); Apply "expandvars()", "normpath()" and "normcase()" to path.
          None value is returned if 'path' is an empty string, or already
          equals None.
          
          param path                  path to convert
          param baseDir               base for absolute path
          return                      either None value or absolute path
        """
        if path is None :
          return None
        elif 0 == len(path) :
          return None
        else :
          result = os.path.expandvars(path)
          result = asqPath.absolute(result, baseDir)
          return asqPath.normalize(result)


    def completePaths(self, baseDir=None) :
        """\
          void completePaths(var baseDir)
          
          Permanently replace task's file paths with absolute versions,
          using 'baseDir' value as base dir (thus the name ;).
          
          param baseDir               base for absolute path
                                        Defaults to None
        """
        asqBuild.logger.trace('task.completePaths(%s)', baseDir)
        asqBuild.logger.incDepthTrace()
        try :
          
          # workdir can not be completed with itself :)
          buffer = self._completePath(self.workdir, baseDir)
          object.__setattr__(self, 'workdir', buffer)
          if baseDir is None :
            baseDir = self.workdir
           
          for key in self.attributes :
            asqBuild.logger.trace('self._attributeIsPath(%s)', key)
            if not self._attributeIsPath(key) :
              pass
            elif self.__dict__[key] is None :
              pass # expecting regular string value
            elif not isinstance(self.__dict__[key], basestring) :
              # replace value with None
              object.__setattr__(self, key, None)
            else :
              # a single path string could actually hold many paths
              # e.g. c:\foo;d:\bar;....
              # for now, this feature is only used with 'delTask' type
              chunks = self.__dict__[key].split(';')
              # applying _completePath to each non-empty / non-null item
              chunks[:] = [ self._completePath(s, baseDir) for s in chunks if s ]
              
              assert 0 == chunks.count(None)                            , \
                'None value found in completed paths list'
              
              buffer = ';'.join(chunks)
              if 0 == len(buffer) :
                object.__setattr__(self, key, None)
              else :
                object.__setattr__(self, key, buffer)
          
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def normalize(self) :
        """\
          void normalize(void)
          
          Corrects values types for some peculiar attributes; e.g. the
          "enabled" attribute must be a boolean value, None value will
          be converted to False
        """
        asqBuild.logger.trace('task.normalize()')
        asqBuild.logger.incDepthTrace()
        try :
          
          for key in self.attributes :
            asqBuild.logger.trace('self._attributeNormalize(%s)', key)
            self._attributeNormalize(key)
          
        finally :
          asqBuild.logger.decDepthTrace()


    def reset(self) :
        """\
          void reset(void)
          
          Set object's attributes to their default values.
        """
        asqBuild.logger.trace('task.reset()')
        asqBuild.logger.incDepthTrace()
        try :
         
          for key in self.attributes :
            asqBuild.logger.trace('self._attributeReset(%s)', key)
            self._attributeReset(key)
         
        finally :
          asqBuild.logger.decDepthTrace()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def beginUpdate(self) :
        """\
          void beginUpdate(void)
          
          Disables automatic attribute "normalization". beginUpdate and
          endUpdate methods should be used when several attributes would
          be set altogether.
          DO NOT FORGET TO USE BOTH METHODS (begin AND end)
        """
        self.__updating += 1


    def endUpdate(self) :
        """\
          void endUpdate(void)
          
          Enables automatic attribute "normalization". beginUpdate and
          endUpdate methods should be used when several attributes would
          be set altogether.
          DO NOT FORGET TO USE BOTH METHODS (begin AND end)
        """
        if 0 < self.__updating :
          self.__updating -= 1


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def isValid(self) :
        """\
          bool isValid(void)
          
          Tells whether or not the task is "valid", id est, if mandatory
          attributes are set; an empty task is not valid, but an invalid
          task may not be empty ;)
          
          return                      is task valid
        """
        asqBuild.logger.trace('task.isValid()')
        asqBuild.logger.incDepthTrace()
        try :
          
          return bool(self)
          
        finally :
          asqBuild.logger.decDepthTrace()


    def isCompatibleWith(self, aType) :
        """\
          bool isCompatibleWith(string aType)
          
          Return whether or not the given 'aType' is compatible with
          the task type.
          
          param aType                 a task type
          return                      task type compatibility
        """
        # no compatibility
        return aType in self.compatibility


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  # /class task - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -