#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

"""\
Defines a checkResult object, which held task checking messages
(hence the name). Inspired by the JUnit implementation (TestResult).

Require
  Python        2.2

Classes
  class         checkResult(void)
"""

__author__  = "Benoit Kogut-Kubiak"
__email__   = "benoit.kogutkubiak@netasq.com"
__version__ = "$Revision: 1.13 $"[11:-2]


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

import  logging

from    py_netasq.commonlib     import asqTypes

from    py_netasq               import building as asqBuild

# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

class checkResult(object) :
    """\
      Custom class, which handles check results, as strings (messages).
      Four message types do exists : information, warning, error and
      critical error.
      Each type could be represented with a single character
      e.g. '!' for warnings, '*' for errors, ....
      
      
      constructor       checkResult(string header, string footer)
      
      void              debug(void)
      
      void              report(object logger)
      
      void              reset(void)
      void              clearMessages(void)
      void              clearChildren(void)
      
      object            appendChild(object child, *arglist, **argdict)
      
      void              addMessage(int type, string value)
      void              addMsgInfo(string value)
      void              addMsgWarning(string value)
      void              addMsgError(string value)
      void              addMsgCritical(string value)
      
      property string   header
      property string   footer
      
      property list     infos
      property list     warnings
      property list     errors
      property list     criticals
      
      property int      totalInfosCount
      property int      totalWarningsCount
      property int      totalErrorsCount
      property int      totalCriticalsCount
    """

    _levels     = (
        logging.INFO,
        logging.CRITICAL,
        logging.ERROR,
        logging.WARNING,
      )

    _labels     = {
        logging.INFO    : 'Info',
        logging.CRITICAL: 'Critical',
        logging.ERROR   : 'Error',
        logging.WARNING : 'Warning',
      }

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __init__(self, header='', footer='.') :
        """\
          constructor checkResult(string header, string footer)
          
          Creates a new checkResult instance. when "report" is called,
          'header' and 'footer' are printed out before and after the
          result's report (if different from empty strings).
          
          param header                result's header, defaults to ''
          param footer                result's footer, defaults to ''
        """
        self._messages = dict()
        self._children = list()
        self.header    = header
        self.footer    = footer
        self.clearMessages()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def __nonzero__(self) :
        """\
          bool __nonzero__(void)
          
          Called to implement truth value testing, and the built-in operation
          bool(); should return False or True, or their integer equivalents
          0 or 1.
          Tells whether or not the task is "empty": an empty task has no
          value defined (None values) for its attributes. Performed right
          after a "task.reset()" this function should return "False".
          
          return                      is task not empty
        """
        for type in self._levels :
          if self._messages[type] :
            return True
        else :
          return False


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def debug(self) :
        """\
          void debug(void)
          
          Outputs attribute values.
        """
        for type in self._levels :
          length = len(self._messages[type])
          asqBuild.logger.debug('%-10s: %u', self._labels[type], length)
          asqBuild.logger.incDepth()
          for key in xrange(length) :
            buffer = repr(self._messages[type][key])
            asqBuild.logger.debug('%03u %s', key, buffer)
          asqBuild.logger.decDepth()
        
        length = len(self._children)
        asqBuild.logger.debug('len(children): %u', length)
        for key in xrange(length) :
          asqBuild.logger.debug('item %03u / %03u', key, length-1)
          asqBuild.logger.incDepth()
          self._children[key].debug()
          asqBuild.logger.decDepth()


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def report(self, display) :
        """\
          void report(object display)
          
          Prints out the report using given logger object 'display' as
          output. Since checkResult's use the same logging levels valus,
          each messages are displayed through the accurate method
          (display.warning(), display.critical(), ...).
          To avoid information cluttering, only non empty checkResults
          are displayed.
          
          param display               logger object
        """
        if self : # not empty
          if asqTypes.isNonEmptyInstance(self._header, basestring) :
            display.info(self._header)
          
          for type in self._levels :
            if 0 == len(self._messages[type]) :
              continue
            else :
              display.info('---')
              display.incDepth()
              for msg in self._messages[type] :
                #~ display.log(type, '%s', msg)
                display.log(type, '%-10s %s', self._labels[type], msg)
              display.decDepth()
          
          if asqTypes.isNonEmptyInstance(self._footer, basestring) :
            display.info(self._footer)
        
        for child in self._children :
          child.report(display)


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def reset(self) :
        """\
          void reset(void)
          
          Empties messages and children lists. Resets header and footer
          to their default values.
        """
        self.clearMessages()
        self.clearChildren()
        self.header = ''
        self.footer = '.'


    def clearMessages(self) :
        """\
          void clearMessages(void)
          
          Recursively clears the check messages, ALL messages items
          are removed.
        """
        for child in self._children :
          child.clearMessages()
        self._messages.clear()
        for type in self._levels :
          self._messages[type] = list()


    def clearChildren(self) :
        """\
          void clearChildren(void)
          
          Recursively clears the checkResult children.
        """
        for key in xrange(len(self._children)) :
          self._children[key].clearChildren()
        self._children[:] = []


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def appendChild(self, child=None, *arglist, **argdict) :
        """\
          object appendChild(object child, *arglist, **argdict)
          
          Appends a new checkResult to the children list, returns the added
          item. If 'child' is None, then a new checkResult will be created
          using the given dynamic arguments (arglist and argdict)
          
          Note : because data duplication is evil, if given 'child' is
           found to already be present in the children list, then it is
           returned and NOT added twice.
          
          param child                 item to append, defaults to None
          param *arglist              dynamic parameters list
          param **argdict             dynamic parameters dict
          return                      added item
        """
        try :
          offset = self._children.index(child)
        except ValueError, e :
          if child is not None :
            result = child
          else :
            result = checkResult(*arglist, **argdict)
          self._children.append(result)
          return result
        else :
          return self._children[offset]


    def addMessage(self, type, value) :
        """\
          void addMessage(int type, string value)
          
          Appends message to the status instance.
          
          param type                  message type
          param value                 message to append
        """
        try :
          self._messages[type].append(value)
        except KeyError, e :
          self._messages[logging.INFO].append(value.lstrip())


    def addMsgInfo(self, value) :
        """\
          void addMsgInfo(string value)
          
          Appends info message to the status instance.
          
          param value                 message to append
        """
        self._messages[logging.INFO].append(value.lstrip())


    def addMsgWarning(self, value) :
        """\
          void addMsgWarning(string value)
          
          Appends warning message to the status instance.
          
          param value                 message to append
        """
        self._messages[logging.WARNING].append(value.lstrip())


    def addMsgError(self, value) :
        """\
          void addMsgError(string value)
          
          Appends error message to the status instance.
          
          param value                 message to append
        """
        self._messages[logging.ERROR].append(value.lstrip())


    def addMsgCritical(self, value) :
        """\
          void addMsgCritical(string value)
          
          Appends critical error message to the status instance.
          
          param value                 message to append
        """
        self._messages[logging.CRITICAL].append(value.lstrip())


  # Getters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  
    def _getHeader(self) :
        """\
          string _getHeader(void)
          
          Getter - header property.
        """
        return self._header


    def _getFooter(self) :
        """\
          string _getFooter(void)
          
          Getter - footer property.
        """
        return self._footer


    def _getInfos(self) :
        """\
          list _getInfos(void)
          
          Getter - infos property.
        """
        return self._messages[logging.INFO]


    def _getWarnings(self) :
        """\
          list _getWarnings(void)
          
          Getter - warnings property.
        """
        return self._messages[logging.WARNING]


    def _getErrors(self) :
        """\
          list _getErrors(void)
          
          Getter - errors property.
        """
        return self._messages[logging.ERROR]


    def _getCriticals(self) :
        """\
          list _getCriticals(void)
          
          Getter - criticals property.
        """
        return self._messages[logging.CRITICAL]


  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _getTotalInfosCount(self) :
        """\
          int _getTotalInfosCount(void)
          
          Getter - totalInfosCount property
          Recursive info messages count
        """
        result = len(self._messages[logging.INFO])
        for child in self._children :
          result += child._getTotalInfosCount()
        return result


    def _getTotalWarningsCount(self) :
        """\
          int _getTotalWarningsCount(void)
          
          Getter - totalWarningsCount property
          Recursive warning messages count
        """
        result = len(self._messages[logging.WARNING])
        for child in self._children :
          result += child._getTotalWarningsCount()
        return result


    def _getTotalErrorsCount(self) :
        """\
          int _getTotalErrorsCount(void)
          
          Getter - totalErrorsCount property
          Recursive error messages count
        """
        result = len(self._messages[logging.ERROR])
        for child in self._children :
          result += child._getTotalErrorsCount()
        return result


    def _getTotalCriticalsCount(self) :
        """\
          int _getTotalCriticalsCount(void)
          
          Getter - totalCriticalsCount property
          Recursive critical messages count
        """
        result = len(self._messages[logging.CRITICAL])
        for child in self._children :
          result += child._getTotalCriticalsCount()
        return result


  # Setters - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    def _setHeader(self, value) :
        """\
          void _setHeader(string value)
          
          Setter - header property.
        """
        if value is None :
          self._header = ''
        else :
          self._header = str(value)


    def _setFooter(self, value) :
        """\
          void _setFooter(string value)
          
          Setter - footer property.
        """
        if value is None :
          self._footer = ''
        else :
          self._footer = str(value)


  # Properties  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    header      = property(
        doc  = "Report's header",
        fget = _getHeader,
        fset = _setHeader,
        fdel = None
      )

    footer      = property(
        doc  = "Report's footer",
        fget = _getFooter,
        fset = _setFooter,
        fdel = None
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    infos       = property(
        doc  = "Info messages list",
        fget = _getInfos,
        fset = None,
        fdel = None
      )
    
    warnings    = property(
        doc  = "Warning messages list",
        fget = _getWarnings,
        fset = None,
        fdel = None
      )
    
    errors      = property(
        doc  = "Error messages list",
        fget = _getErrors,
        fset = None,
        fdel = None
      )
    
    criticals   = property(
        doc  = "Critical messages list",
        fget = _getCriticals,
        fset = None,
        fdel = None
      )

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

    totalInfosCount     = property(
        doc  = "Recursive info messages count",
        fget = _getTotalInfosCount,
        fset = None,
        fdel = None
      )

    totalWarningsCount  = property(
        doc  = "Recursive warning messages count",
        fget = _getTotalWarningsCount,
        fset = None,
        fdel = None
      )

    totalErrorsCount    = property(
        doc  = "Recursive error messages count",
        fget = _getTotalErrorsCount,
        fset = None,
        fdel = None
      )

    totalCriticalsCount = property(
        doc  = "Recursive critical messages count",
        fget = _getTotalCriticalsCount,
        fset = None,
        fdel = None
      )

  # /class checkResult  - - - - - - - - - - - - - - - - - - - - - - - - - - - -



# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

if '__main__' == __name__ :
  print __file__
  print __doc__


# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -